class preferencesPlugin extends BasePlugin {
    hotkey = () => [{ hotkey: this.config.HOTKEY, callback: this.call }]

    styleTemplate = () => true

    html = () => `
        <div id="plugin-preferences-dialog" class="plugin-common-hidden">
            <div id="plugin-preferences-dialog-content">
                <div id="plugin-preferences-dialog-left">
                    <div id="plugin-preferences-menu-search">
                        <input type="text" placeholder="${this.i18n._t("global", "search")}">
                    </div>
                    <div id="plugin-preferences-menu"></div>
                </div>
                <div id="plugin-preferences-dialog-right">
                    <div id="plugin-preferences-dialog-title"></div>
                    <div id="plugin-preferences-dialog-close" class="ion-close-round"></div>
                    <div id="plugin-preferences-dialog-main">
                        <dialog-form id="plugin-preferences-dialog-form" data-plugin="global"></dialog-form>
                    </div>
                </div>
            </div>
        </div>
    `

    init = () => {
        this.entities = {
            dialog: document.querySelector("#plugin-preferences-dialog"),
            menu: document.querySelector("#plugin-preferences-menu"),
            title: document.querySelector("#plugin-preferences-dialog-title"),
            form: document.querySelector("#plugin-preferences-dialog-form"),
            main: document.querySelector("#plugin-preferences-dialog-main"),
            searchInput: document.querySelector("#plugin-preferences-menu-search input"),
            closeButton: document.querySelector("#plugin-preferences-dialog-close"),
        }
        this._initActionHandlers()
        this._initPreProcessors()
        this._initPostProcessors()
        this._initSchemas()
        this._initDialogForm()
    }

    process = () => {
        const searchInDialog = () => {
            let allow = true
            const search = () => {
                if (!allow) return
                const query = this.entities.searchInput.value.trim().toLowerCase()
                this.entities.menu.querySelectorAll(".plugin-preferences-menu-item").forEach((el) => {
                    let fn = "show"
                    if (query) {
                        const hitShowName = el.textContent.toLowerCase().includes(query)
                        const hitFixedName = this.config.SEARCH_PLUGIN_FIXEDNAME && el.dataset.plugin.toLowerCase().includes(query)
                        if (!hitShowName && !hitFixedName) {
                            fn = "hide"
                        }
                    }
                    this.utils[fn](el)
                })
                if (!query) {
                    const active = this.entities.menu.querySelector(".plugin-preferences-menu-item.active")
                    if (active) {
                        active.scrollIntoView({ block: "center" })
                    }
                }
            }
            this.entities.searchInput.addEventListener("input", search)
            this.entities.searchInput.addEventListener("compositionstart", () => allow = false)
            this.entities.searchInput.addEventListener("compositionend", () => {
                allow = true
                search()
            })
        }
        const domEvents = () => {
            // this.utils.dragFixedModal(this.entities.title, this.entities.dialog, false)
            this.entities.closeButton.addEventListener("click", () => {
                this.call()
                this.utils.notification.show(this.i18n._t("global", "takesEffectAfterRestart"))
            })
            this.entities.menu.addEventListener("click", async ev => {
                const target = ev.target.closest(".plugin-preferences-menu-item")
                if (target) {
                    const fixedName = target.dataset.plugin
                    await this.switchMenu(fixedName)
                }
            })
        }
        const formEvents = () => {
            this.entities.form.addEventListener("CRUD", async ev => {
                const { key, value, type } = ev.detail
                const propHandler = this.utils.nestedPropertyHelpers[type]
                if (propHandler) {
                    const fixedName = this.entities.form.dataset.plugin
                    const settings = await this._getSettings(fixedName)
                    propHandler(settings, key, value)
                    await this.utils.settings.saveSettings(fixedName, settings)
                    const postFn = this.POSTPROCESSORS[`${fixedName}.${key}`]
                    if (postFn) {
                        await postFn(value, settings)
                    }
                    return
                }
                const actionHandler = type === "action" && this.ACTION_HANDLERS[key]
                if (actionHandler) {
                    await actionHandler()
                }
            })
        }

        searchInDialog()
        domEvents()
        formEvents()
    }

    call = async () => {
        const isShow = this.utils.isShow(this.entities.dialog)
        if (isShow) {
            this.entities.searchInput.value = ""
            this.utils.hide(this.entities.dialog)
        } else {
            await this.showDialog(this.config.DEFAULT_MENU)
            this.utils.show(this.entities.dialog)
        }
    }

    showDialog = async (showMenu) => {
        const plugins = this._getAllPlugins()
        const menus = Object.entries(plugins).map(([name, pluginName]) => {
            const showName = this.utils.escape(pluginName)
            return `<div class="plugin-preferences-menu-item" data-plugin="${name}">${showName}</div>`
        })
        this.entities.menu.innerHTML = menus.join("")
        showMenu = plugins.hasOwnProperty(showMenu) ? showMenu : "global"
        await this.switchMenu(showMenu)
        setTimeout(() => {
            const active = this.entities.menu.querySelector(".plugin-preferences-menu-item.active")
            active.scrollIntoView({ block: "center" })
        }, 50)
    }

    switchMenu = async (fixedName) => {
        const settings = await this._getSettings(fixedName)
        const data = await this._preprocess(fixedName, settings)
        this.entities.form.dataset.plugin = fixedName
        this.entities.form.render(this.SETTING_SCHEMAS[fixedName], data)
        this.entities.menu.querySelectorAll(".active").forEach(e => e.classList.remove("active"))
        const menuItem = this.entities.menu.querySelector(`.plugin-preferences-menu-item[data-plugin="${fixedName}"]`)
        menuItem.classList.add("active")
        this.entities.title.textContent = menuItem.textContent
        $(this.entities.main).animate({ scrollTop: 0 }, 300)
    }

    _getAllPlugins = () => {
        const names = [
            "global",
            ...Object.keys(this.utils.getAllPluginSettings()),
            ...Object.keys(this.utils.getAllCustomPluginSettings())
        ]
        const plugins = names
            .filter(name => this.SETTING_SCHEMAS.hasOwnProperty(name))
            .map(name => {
                const p = this.utils.tryGetPlugin(name)
                const pluginName = p ? p.pluginName : this.i18n._t(name, "pluginName")
                return [name, pluginName]
            })
        return Object.fromEntries(plugins)
    }

    _getSettings = async (fixedName) => {
        const isBase = this.utils.getPluginSetting(fixedName)
        const fn = isBase ? "readBasePluginSettings" : "readCustomPluginSettings"
        const settings = await this.utils.settings[fn]()
        return settings[fixedName]
    }

    _preprocess = async (fixedName, settings) => {
        const fnMap = this.PREPROCESSORS
        await Promise.all(
            this.SETTING_SCHEMAS[fixedName].flatMap(box => {
                return box.fields
                    .filter(field => field.key && fnMap.hasOwnProperty(`${fixedName}.${field.key}`))
                    .map(async field => await fnMap[`${fixedName}.${field.key}`](field, settings))
            })
        )
        return settings
    }

    _initDialogForm = () => this.entities.form.init(this.utils, { objectFormat: this.config.OBJECT_SETTINGS_FORMAT })

    /** Will NOT modify the schemas structure, just i18n */
    _translateSchema = (schemas) => {
        const specialProps = ["options", "thMap"]
        const baseProps = ["label", "tooltip", "placeholder", "hintHeader", "hintDetail"]
        const commonProps = [...baseProps, "title", "unit"]

        const i18nData = this.i18n.noConflict.data
        const commonI18N = Object.fromEntries(
            commonProps.map(prop => {
                const val = this.utils.pickBy(i18nData.settings, (val, key) => key.startsWith(`$${prop}.`))
                return [prop, val]
            })
        )

        const translateFieldBaseProps = (field, pluginI18N) => {
            baseProps.forEach(prop => {
                const propVal = field[prop]
                if (propVal != null) {
                    const commonVal = commonI18N[prop][propVal]
                    const pluginVal = pluginI18N[propVal]
                    field[prop] = commonVal || pluginVal
                }
            })
        }
        const translateFieldSpecialProps = (field, pluginI18N) => {
            specialProps.forEach(prop => {
                const propVal = field[prop]
                if (propVal != null) {
                    Object.keys(propVal).forEach(k => {
                        const i18nKey = propVal[k]
                        propVal[k] = pluginI18N[i18nKey]
                    })
                }
            })
        }
        const translateFieldNestedBoxesProp = (field, pluginI18N) => {
            if (field.nestedBoxes != null) {
                field.nestedBoxes.forEach(box => translateBox(box, pluginI18N))
            }
        }
        const translateFieldUnitProp = (field) => {
            if (field.unit != null) {
                field.unit = commonI18N.unit[field.unit]
            }
        }
        const translateBox = (box, pluginI18N) => {
            const t = box.title
            if (t) {
                const commonVal = commonI18N.title[t]
                const pluginVal = pluginI18N[t]
                box.title = commonVal || pluginVal
            }
            box.fields.forEach(field => {
                translateFieldBaseProps(field, pluginI18N)
                translateFieldSpecialProps(field, pluginI18N)
                translateFieldNestedBoxesProp(field, pluginI18N)
                translateFieldUnitProp(field)
            })
        }

        Object.entries(schemas).forEach(([fixedName, boxes]) => {
            const pluginI18N = i18nData[fixedName]
            boxes.forEach(box => translateBox(box, pluginI18N))
        })
    }

    _removeDependencies = (obj) => {
        if (obj == null || typeof obj !== "object") return

        for (const key in obj) {
            if (obj.hasOwnProperty(key)) {
                if (key === "dependencies") {
                    delete obj[key]
                } else if (typeof obj[key] === "object") {
                    this._removeDependencies(obj[key])
                }
            }
        }
    }

    _initSchemas = () => {
        this.SETTING_SCHEMAS = require("./schemas.js")

        this._translateSchema(this.SETTING_SCHEMAS)
        if (this.config.IGNORE_CONFIG_DEPENDENCIES) {
            this._removeDependencies(this.SETTING_SCHEMAS)
        }
    }

    /** Callback functions for type="action" settings in schema */
    _initActionHandlers = () => {
        this.ACTION_HANDLERS = {
            visitRepo: () => this.utils.openUrl("https://github.com/obgnail/typora_plugin"),
            deepWiki: () => this.utils.openUrl("https://deepwiki.com/obgnail/typora_plugin"),
            githubImageBed: () => this.utils.openUrl("https://github.com/obgnail/typora_image_uploader"),
            sendEmail: () => this.utils.sendEmail("he1251698542@gmail.com", "Feedback"),
            viewMarkdownlintRules: () => this.utils.openUrl("https://github.com/DavidAnson/markdownlint/blob/main/doc/Rules.md"),
            viewCustomMarkdownlintRules: () => this.utils.openUrl("https://github.com/obgnail/markdownlint-rule-math"),
            viewCodeMirrorKeymapsManual: () => this.utils.openUrl("https://codemirror.net/5/doc/manual.html#keymaps"),
            viewAbcVisualOptionsHelp: () => this.utils.openUrl("https://paulrosen.github.io/abcjs/visual/render-abc-options.html"),
            chooseEchartsRenderer: () => this.utils.openUrl("https://echarts.apache.org/handbook/en/best-practices/canvas-vs-svg/"),
            viewArticleUploaderReadme: () => this.utils.showInFinder(this.utils.joinPath("./plugin/article_uploader/README.md")),
            viewJsonRPCReadme: () => this.utils.showInFinder(this.utils.joinPath("./plugin/json_rpc/README.md")),
            editStyles: () => this.utils.showInFinder(this.utils.joinPath("./plugin/global/user_styles/README.md")),
            developPlugins: () => this.utils.showInFinder(this.utils.joinPath("./plugin/custom/README.md")),
            backupSettings: async () => this.utils.settings.backupSettingFile(),
            openSettingsFolder: async () => this.utils.settings.openSettingFolder(),
            restoreSettings: async () => {
                const fixedName = this.entities.form.dataset.plugin
                await this.utils.settings.clearSettings(fixedName)
                await this.switchMenu(fixedName)
                this.utils.notification.show(this.i18n._t("global", "notification.settingsRestored"))
            },
            restoreAllSettings: async () => {
                const fixedName = this.entities.form.dataset.plugin
                await this.utils.settings.clearAllSettings()
                await this.switchMenu(fixedName)
                this.utils.notification.show(this.i18n._t("global", "notification.allSettingsRestored"))
            },
            runtimeSettings: async () => {
                const fixedName = this.entities.form.dataset.plugin
                const settings = await this._getSettings(fixedName)
                const op = {
                    title: this.i18n._t("settings", "$label.runtimeSettings") + `（${this.i18n._t("global", "readonly")}）`,
                    schema: [{ fields: [{ key: "runtimeSettings", type: "textarea", rows: 14 }] }],
                    data: { runtimeSettings: JSON.stringify(settings, null, "\t") },
                }
                await this.utils.formDialog.modal(op)
            },
            updatePlugin: async () => {
                const updater = this.utils.getPlugin("updater")
                if (!updater) {
                    const plugin = this.i18n._t("updater", "pluginName")
                    const msg = this.i18n._t("global", "error.pluginDisabled", { plugin })
                    this.utils.notification.show(msg, "error")
                } else {
                    await updater.call()
                }
            },
            uninstallPlugin: async () => {
                const uninstall = async () => {
                    const { FsExtra } = this.utils.Package
                    const remove = '<script src="./plugin/index.js" defer="defer"></script>'
                    const windowHTML = this.utils.joinPath("./window.html")
                    const pluginFolder = this.utils.joinPath("./plugin")
                    try {
                        const content = await FsExtra.readFile(windowHTML, "utf-8")
                        const newContent = content.replace(remove, "")
                        await FsExtra.writeFile(windowHTML, newContent)
                        await FsExtra.remove(pluginFolder)
                    } catch (e) {
                        alert(e.toString())
                        return
                    }
                    const message = this.i18n._t("global", "uninstallPluginSuccessful")
                    const confirm = this.i18n._t("global", "confirm")
                    const op = { type: "info", title: "typora plugin", message, buttons: [confirm] }
                    await this.utils.showMessageBox(op)
                    this.utils.restartTypora(false)
                }

                const title = this.i18n._t("global", "$label.uninstallPlugin")
                const hintHeader = this.i18n._t("global", "uninstallPluginWarning")
                const hintDetail = this.i18n._t("global", "uninstallPluginDetail", { reconfirm: title })
                const label = this.i18n._t("global", "uninstallPluginConfirmInput")
                const op = {
                    title,
                    schema: [
                        { fields: [{ type: "hint", hintHeader, hintDetail }] },
                        { fields: [{ type: "text", key: "confirmInput", label, placeholder: title }] },
                    ],
                    data: { confirmInput: "" },
                }
                const { response, data } = await this.utils.formDialog.modal(op)
                if (response === 0) return
                if (data.confirmInput !== title) {
                    const msg = this.i18n._t("global", "error.incorrectCommand")
                    this.utils.notification.show(msg, "error")
                } else {
                    await uninstall()
                }
            },
            donate: () => {
                const weChatPay = "8-RWSVREYNE9TCVADDKEGVPNJ1KGAYNZ31KENF2LWDEA3KFHHDRWYEPA4F00KSZT3454M24RD5PVVM21AAJ5DAGMQ3H62CHEQOOT226D49LZR6G1FKOG0G7NUV5GR2HD2B6V3V8DHR2S8027S36ESCU3GJ0IAE7IY9S25URTMZQCZBY8ZTHFTQ45VVGFX3VD1SE9K4Y9K7I1Y7U4FIKZSS2Y87BH4OSASYLS48A6SR2T5YZJNMJ2WCQE0ZBK9OVLGWGWGL1ED400U1BYMZRW7UAS7VECNVL98WKG4PNIF0KFNIVS45KHQXJFH9E9SYRCWYRUX45Q37"
                const aliPay = "9-CF07WK7ZZ6CKLVC5KX92LZGUL3X93E51RYAL92NHYVQSD6CAH4D1DTCENAJ8HHB0062DU7LS29Q8Y0NT50M8XPFP9N1QE1JPFW39U0CDP2UX9H2WLEYD712FI3C5657LIWMT7K5CCVL509G04FT4N0IJD3KRAVBDM76CWI81XY77LLSI2AZ668748L62IC4E8CYYVNBG4Z525HZ4BXQVV6S81JC0CVABEACU597FNP9OHNC959X4D29MMYXS1V5MWEU8XC4BD5WSLL29VSAQOGLBWAVVTMX75DOSRF78P9LARIJ7J50IK1MM2QT5UXU5Q1YA7J2AVVHMG00E06Q80RCDXVGOFO76D1HCGYKW93MXR5X4H932TYXAXL93BYWV9UH6CTDUDFWACE5G0OM9N"
                const qrcodeList = [{ color: "#1AAD19", compressed: weChatPay }, { color: "#027AFF", compressed: aliPay }]
                const size = 140
                const margin = 60
                const backgroundColor = "#F3F2EE"
                const canvasWidth = (size + margin) * qrcodeList.length - margin

                const _decompress = (compressed) => {
                    const [chunk, raw] = compressed.split("-", 2)
                    const rows = raw.match(new RegExp(`\\w{${chunk}}`, "g"))
                    return rows.map(r => parseInt(r, 36).toString(2).padStart(rows.length, "0"))
                }
                const _adaptDPR = (canvas, ctx) => {
                    const dpr = File.canvasratio || window.devicePixelRatio || 1
                    const { width, height } = canvas
                    canvas.width = Math.round(width * dpr)
                    canvas.height = Math.round(height * dpr)
                    canvas.style.width = width + "px"
                    canvas.style.height = height + "px"
                    ctx.scale(dpr, dpr)
                }
                const onload = (dialog = document) => {
                    const canvas = dialog.querySelector("canvas")
                    if (!canvas) return

                    const ctx = canvas.getContext("2d")
                    _adaptDPR(canvas, ctx)
                    ctx.lineWidth = 0
                    ctx.strokeStyle = "transparent"
                    for (const { compressed, color } of qrcodeList) {
                        ctx.fillStyle = backgroundColor
                        ctx.fillRect(0, 0, size, size)
                        ctx.fillStyle = color
                        const table = _decompress(compressed)
                        const rectWidth = size / table.length
                        // Division and canvas pixel magnification issues lead to precision loss. Adding 0.3 makes it look better.
                        const rectWidth2 = rectWidth + 0.3
                        for (let cIdx = 0; cIdx < table.length; cIdx++) {
                            for (let rIdx = 0; rIdx < table[0].length; rIdx++) {
                                if (table[cIdx][rIdx] === "1") {
                                    ctx.fillRect(rIdx * rectWidth, cIdx * rectWidth, rectWidth2, rectWidth2)
                                }
                            }
                        }
                        ctx.translate(size + margin, 0)
                    }
                }
                const canvas = `<canvas width="${canvasWidth}" height="${size}" style="margin: auto; display: block;"></canvas>`
                const title = this.i18n._t("global", "$label.donate")
                const components = [{ label: "", type: "span" }, { label: canvas, type: "span" }]
                const op = { title, components, onload, width: "500px" }
                this.utils.dialog.modal(op)
            },
        }
    }

    /** PreProcessors for specific settings in schema */
    _initPreProcessors = () => {
        const _disableOption = (options, targetOption) => Object.defineProperty(options, targetOption, { enumerable: false })
        const _incompatibleSwitch = (field, settings, tooltip = this.i18n._t("settings", "$tooltip.lowVersion")) => {
            field.disabled = true
            field.tooltip = tooltip
            settings[field.key] = false
        }
        this.PREPROCESSORS = {
            "global.pluginVersion": async (field, data) => {
                if (!data[field.key]) {
                    let version = "Unknown"
                    try {
                        const file = this.utils.joinPath("./plugin/bin/version.json")
                        const json = await this.utils.Package.FsExtra.readJson(file)
                        version = json.tag_name
                    } catch (e) {
                        console.error(e)
                    }
                    data[field.key] = version
                }
            },
            "window_tab.LAST_TAB_CLOSE_ACTION": (field, data) => {
                if (this.utils.isBetaVersion) {
                    const illegalOption = "blankPage"
                    _disableOption(field.options, illegalOption)
                    if (data[field.key] === illegalOption) {
                        data[field.key] = "reconfirm"
                    }
                }
            },
            "fence_enhance.ENABLE_INDENT": (field, data) => {
                if (this.utils.isBetaVersion) {
                    _incompatibleSwitch(field, data)
                }
            },
            "blur.ENABLE": (field, data) => {
                if (!this.utils.supportHasSelector) {
                    _incompatibleSwitch(field, data)
                }
            },
            "export_enhance.ENABLE": (field, data) => {
                if (!this.utils.exportHelper.isAsync) {
                    _incompatibleSwitch(field, data)
                }
            },
            "markmap.AUTO_COLLAPSE_PARAGRAPH_WHEN_FOLD": (field, data) => {
                if (!this.utils.getPlugin("collapse_paragraph")) {
                    _incompatibleSwitch(field, data, this.i18n._t("markmap", "$tooltip.experimental"))
                }
            },
            "reopenClosedFiles.enable": (field, data) => {
                if (!this.utils.getPlugin("window_tab")) {
                    _incompatibleSwitch(field, data, this.i18n._t("reopenClosedFiles", "$tooltip.dependOnWindowTab"))
                }
            },
            "preferences.DEFAULT_MENU": (field, data) => {
                if (!field.options) {
                    field.options = this._getAllPlugins()
                }
            },
        }
    }

    /** PostProcessors for specific settings in schema */
    _initPostProcessors = () => {
        this.POSTPROCESSORS = {}
    }
}

module.exports = {
    plugin: preferencesPlugin
}
